<?php
session_start();
require_once 'config/database.php';
require_once 'includes/auth.php';
require_once 'includes/security.php';

$auth = new Auth();
$auth->requireAdmin();

$security = new Security();
$csrf_token = $security->generateCSRFToken();

$db = new Database();

// Handle user actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $security->verifyCSRFToken();
    
    $action = $_POST['action'] ?? '';
    $user_id = $_POST['user_id'] ?? '';
    
    switch ($action) {
        case 'toggle_status':
            toggleUserStatus($db, $user_id, $security);
            break;
        case 'delete_user':
            deleteUser($db, $user_id, $security);
            break;
        case 'reset_attempts':
            resetFailedAttempts($db, $user_id, $security);
            break;
        case 'create_user':
            createUser($db, $security);
            break;
        case 'update_user':
            updateUser($db, $user_id, $security);
            break;
    }
}

function toggleUserStatus($db, $user_id, $security) {
    if ($user_id == $_SESSION['admin_id']) {
        $_SESSION['error'] = 'Cannot deactivate your own account';
        return;
    }
    
    $sql = "UPDATE admin_users SET is_active = CASE WHEN is_active = 1 THEN 0 ELSE 1 END WHERE id = ?";
    $params = [$user_id];
    $stmt = $db->prepare($sql, $params);
    
    if ($stmt && $db->execute($stmt)) {
        $security->logActivity($_SESSION['admin_id'], 'USER_STATUS_CHANGED', "Toggled status for user ID: $user_id");
        $_SESSION['message'] = 'User status updated successfully';
    } else {
        $_SESSION['error'] = 'Failed to update user status';
    }
}

function deleteUser($db, $user_id, $security) {
    if ($user_id == $_SESSION['admin_id']) {
        $_SESSION['error'] = 'Cannot delete your own account';
        return;
    }
    
    $sql = "DELETE FROM admin_users WHERE id = ?";
    $params = [$user_id];
    $stmt = $db->prepare($sql, $params);
    
    if ($stmt && $db->execute($stmt)) {
        $security->logActivity($_SESSION['admin_id'], 'USER_DELETED', "Deleted user ID: $user_id");
        $_SESSION['message'] = 'User deleted successfully';
    } else {
        $_SESSION['error'] = 'Failed to delete user';
    }
}

function resetFailedAttempts($db, $user_id, $security) {
    $sql = "UPDATE admin_users SET failed_attempts = 0 WHERE id = ?";
    $params = [$user_id];
    $stmt = $db->prepare($sql, $params);
    
    if ($stmt && $db->execute($stmt)) {
        $security->logActivity($_SESSION['admin_id'], 'RESET_ATTEMPTS', "Reset failed attempts for user ID: $user_id");
        $_SESSION['message'] = 'Failed attempts reset successfully';
    } else {
        $_SESSION['error'] = 'Failed to reset attempts';
    }
}

function createUser($db, $security) {
    $full_name = trim($_POST['full_name']);
    $username = trim($_POST['username']);
    $email = trim($_POST['email']);
    $role = $_POST['role'];
    $password = $_POST['password'];
    
    // Validate inputs
    if (empty($full_name) || empty($username) || empty($email) || empty($password)) {
        $_SESSION['error'] = 'All fields are required';
        return;
    }
    
    // Check if username already exists
    $check_sql = "SELECT id FROM admin_users WHERE username = ?";
    $check_stmt = $db->prepare($check_sql, [$username]);
    if ($db->execute($check_stmt) && $db->fetchArray($check_stmt)) {
        $_SESSION['error'] = 'Username already exists';
        return;
    }
    
    $password_hash = password_hash($password, PASSWORD_DEFAULT);
    
    $sql = "INSERT INTO admin_users (full_name, username, email, role, password_hash) VALUES (?, ?, ?, ?, ?)";
    $params = [$full_name, $username, $email, $role, $password_hash];
    $stmt = $db->prepare($sql, $params);
    
    if ($stmt && $db->execute($stmt)) {
        $new_user_id = $db->getLastInsertId();
        $security->logActivity($_SESSION['admin_id'], 'USER_CREATED', "Created new user: $username");
        $_SESSION['message'] = 'User created successfully';
    } else {
        $_SESSION['error'] = 'Failed to create user';
    }
}

function updateUser($db, $user_id, $security) {
    $full_name = trim($_POST['full_name']);
    $username = trim($_POST['username']);
    $email = trim($_POST['email']);
    $role = $_POST['role'];
    $password = $_POST['password'] ?? '';
    
    // Check if username already exists (excluding current user)
    $check_sql = "SELECT id FROM admin_users WHERE username = ? AND id != ?";
    $check_stmt = $db->prepare($check_sql, [$username, $user_id]);
    if ($db->execute($check_stmt) && $db->fetchArray($check_stmt)) {
        $_SESSION['error'] = 'Username already exists';
        return;
    }
    
    if (!empty($password)) {
        $password_hash = password_hash($password, PASSWORD_DEFAULT);
        $sql = "UPDATE admin_users SET full_name = ?, username = ?, email = ?, role = ?, password_hash = ? WHERE id = ?";
        $params = [$full_name, $username, $email, $role, $password_hash, $user_id];
    } else {
        $sql = "UPDATE admin_users SET full_name = ?, username = ?, email = ?, role = ? WHERE id = ?";
        $params = [$full_name, $username, $email, $role, $user_id];
    }
    
    $stmt = $db->prepare($sql, $params);
    
    if ($stmt && $db->execute($stmt)) {
        $security->logActivity($_SESSION['admin_id'], 'USER_UPDATED', "Updated user: $username");
        $_SESSION['message'] = 'User updated successfully';
    } else {
        $_SESSION['error'] = 'Failed to update user';
    }
}

// Get user details for editing
$edit_user = null;
if (isset($_GET['edit'])) {
    $sql = "SELECT * FROM admin_users WHERE id = ?";
    $stmt = $db->prepare($sql, [$_GET['edit']]);
    if ($db->execute($stmt)) {
        $edit_user = $db->fetchArray($stmt);
    }
}

// Get all users with search and filter
$search = $_GET['search'] ?? '';
$role_filter = $_GET['role'] ?? '';
$status_filter = $_GET['status'] ?? '';

function getUsers($db, $search = '', $role_filter = '', $status_filter = '') {
    $users = [];
    
    $sql = "SELECT id, username, email, full_name, role, is_active, failed_attempts, 
                   last_login, created_at 
            FROM admin_users 
            WHERE 1=1";
    
    $params = [];
    
    if (!empty($search)) {
        $sql .= " AND (username LIKE ? OR full_name LIKE ? OR email LIKE ?)";
        $search_term = "%$search%";
        $params[] = $search_term;
        $params[] = $search_term;
        $params[] = $search_term;
    }
    
    if (!empty($role_filter)) {
        $sql .= " AND role = ?";
        $params[] = $role_filter;
    }
    
    if (!empty($status_filter)) {
        $sql .= " AND is_active = ?";
        $params[] = ($status_filter === 'active') ? 1 : 0;
    }
    
    $sql .= " ORDER BY created_at DESC";
    
    $stmt = $db->prepare($sql, $params);
    
    if ($db->execute($stmt)) {
        while ($row = $db->fetchArray($stmt)) {
            $users[] = $row;
        }
    }
    
    return $users;
}

$users = getUsers($db, $search, $role_filter, $status_filter);

// Get user statistics
function getUserStats($db) {
    $stats = [
        'total' => 0,
        'active' => 0,
        'inactive' => 0,
        'super_admins' => 0,
        'admins' => 0,
        'moderators' => 0,
        'locked' => 0
    ];
    
    $sql = "SELECT 
        COUNT(*) as total,
        SUM(CASE WHEN is_active = 1 THEN 1 ELSE 0 END) as active,
        SUM(CASE WHEN is_active = 0 THEN 1 ELSE 0 END) as inactive,
        SUM(CASE WHEN role = 'super_admin' THEN 1 ELSE 0 END) as super_admins,
        SUM(CASE WHEN role = 'admin' THEN 1 ELSE 0 END) as admins,
        SUM(CASE WHEN role = 'moderator' THEN 1 ELSE 0 END) as moderators,
        SUM(CASE WHEN failed_attempts >= 5 THEN 1 ELSE 0 END) as locked
    FROM admin_users";
    
    $stmt = $db->prepare($sql);
    if ($db->execute($stmt) && $row = $db->fetchArray($stmt)) {
        $stats = $row;
    }
    
    return $stats;
}

$user_stats = getUserStats($db);
?>
<div class="container-fluid px-4">
    <!-- Page Header -->
    <div class="d-flex justify-content-between align-items-center py-3 mb-4 border-bottom">
        <div>
            <h1 class="h3 mb-1">
                <i class="fas fa-users-cog me-2 text-primary"></i>
                User Management
            </h1>
            <p class="text-muted mb-0">Manage system administrators and their permissions</p>
        </div>
        <div class="btn-toolbar">
            <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addUserModal">
                <i class="fas fa-user-plus me-2"></i>
                Add User
            </button>
        </div>
    </div>

    <!-- Messages -->
    <?php if (isset($_SESSION['message'])): ?>
        <div class="alert alert-success alert-dismissible fade show mb-4" role="alert">
            <div class="d-flex align-items-center">
                <i class="fas fa-check-circle me-2"></i>
                <span><?php echo $_SESSION['message']; unset($_SESSION['message']); ?></span>
            </div>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger alert-dismissible fade show mb-4" role="alert">
            <div class="d-flex align-items-center">
                <i class="fas fa-exclamation-circle me-2"></i>
                <span><?php echo $_SESSION['error']; unset($_SESSION['error']); ?></span>
            </div>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
    <?php endif; ?>

    <!-- User Statistics -->
    <div class="row mb-4">
        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
            <div class="card border-0 bg-primary bg-opacity-10">
                <div class="card-body">
                    <div class="d-flex align-items-center">
                        <div class="flex-grow-1">
                            <h4 class="mb-0"><?php echo $user_stats['total']; ?></h4>
                            <span class="text-muted small">Total Users</span>
                        </div>
                        <div class="flex-shrink-0">
                            <i class="fas fa-users fa-2x text-primary opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
            <div class="card border-0 bg-success bg-opacity-10">
                <div class="card-body">
                    <div class="d-flex align-items-center">
                        <div class="flex-grow-1">
                            <h4 class="mb-0"><?php echo $user_stats['active']; ?></h4>
                            <span class="text-muted small">Active</span>
                        </div>
                        <div class="flex-shrink-0">
                            <i class="fas fa-user-check fa-2x text-success opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
            <div class="card border-0 bg-warning bg-opacity-10">
                <div class="card-body">
                    <div class="d-flex align-items-center">
                        <div class="flex-grow-1">
                            <h4 class="mb-0"><?php echo $user_stats['super_admins']; ?></h4>
                            <span class="text-muted small">Super Admins</span>
                        </div>
                        <div class="flex-shrink-0">
                            <i class="fas fa-crown fa-2x text-warning opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
            <div class="card border-0 bg-info bg-opacity-10">
                <div class="card-body">
                    <div class="d-flex align-items-center">
                        <div class="flex-grow-1">
                            <h4 class="mb-0"><?php echo $user_stats['admins']; ?></h4>
                            <span class="text-muted small">Admins</span>
                        </div>
                        <div class="flex-shrink-0">
                            <i class="fas fa-user-shield fa-2x text-info opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
            <div class="card border-0 bg-secondary bg-opacity-10">
                <div class="card-body">
                    <div class="d-flex align-items-center">
                        <div class="flex-grow-1">
                            <h4 class="mb-0"><?php echo $user_stats['moderators']; ?></h4>
                            <span class="text-muted small">Moderators</span>
                        </div>
                        <div class="flex-shrink-0">
                            <i class="fas fa-user-gear fa-2x text-secondary opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <div class="col-xl-2 col-md-4 col-sm-6 mb-3">
            <div class="card border-0 bg-danger bg-opacity-10">
                <div class="card-body">
                    <div class="d-flex align-items-center">
                        <div class="flex-grow-1">
                            <h4 class="mb-0"><?php echo $user_stats['locked']; ?></h4>
                            <span class="text-muted small">Locked</span>
                        </div>
                        <div class="flex-shrink-0">
                            <i class="fas fa-lock fa-2x text-danger opacity-50"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Filters and Search -->
    <div class="card mb-4">
        <div class="card-header bg-transparent">
            <h5 class="mb-0">
                <i class="fas fa-filter me-2 text-muted"></i>
                Filters & Search
            </h5>
        </div>
        <div class="card-body">
            <form method="GET" class="row g-3">
                <div class="col-md-4">
                    <label class="form-label">Search Users</label>
                    <div class="input-group">
                        <span class="input-group-text">
                            <i class="fas fa-search"></i>
                        </span>
                        <input type="text" class="form-control" name="search" placeholder="Search by name, username, or email..." value="<?php echo htmlspecialchars($search); ?>">
                    </div>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Role</label>
                    <select class="form-select" name="role">
                        <option value="">All Roles</option>
                        <option value="super_admin" <?php echo $role_filter === 'super_admin' ? 'selected' : ''; ?>>Super Admin</option>
                        <option value="admin" <?php echo $role_filter === 'admin' ? 'selected' : ''; ?>>Admin</option>
                        <option value="moderator" <?php echo $role_filter === 'moderator' ? 'selected' : ''; ?>>Moderator</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Status</label>
                    <select class="form-select" name="status">
                        <option value="">All Status</option>
                        <option value="active" <?php echo $status_filter === 'active' ? 'selected' : ''; ?>>Active</option>
                        <option value="inactive" <?php echo $status_filter === 'inactive' ? 'selected' : ''; ?>>Inactive</option>
                    </select>
                </div>
                <div class="col-md-2 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary w-100">
                        <i class="fas fa-filter me-1"></i>
                        Apply
                    </button>
                </div>
            </form>
        </div>
    </div>

    <!-- Users Table -->
    <div class="card">
        <div class="card-header bg-transparent d-flex justify-content-between align-items-center">
            <h5 class="mb-0">
                <i class="fas fa-list me-2 text-muted"></i>
                User List
                <span class="badge bg-primary ms-2"><?php echo count($users); ?> users</span>
            </h5>
            <div class="btn-group">
                <button class="btn btn-outline-secondary btn-sm" onclick="window.print()">
                    <i class="fas fa-print me-1"></i>Print
                </button>
                <button class="btn btn-outline-secondary btn-sm">
                    <i class="fas fa-download me-1"></i>Export
                </button>
            </div>
        </div>
        <div class="card-body p-0">
            <div class="table-responsive">
                <table class="table table-hover mb-0">
                    <thead class="table-light">
                        <tr>
                            <th class="ps-4">User</th>
                            <th>Contact</th>
                            <th>Role</th>
                            <th>Status</th>
                            <th>Last Login</th>
                            <th>Created</th>
                            <th class="text-end pe-4">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($users)): ?>
                            <tr>
                                <td colspan="7" class="text-center py-4 text-muted">
                                    <i class="fas fa-users fa-2x mb-3 d-block"></i>
                                    No users found matching your criteria
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($users as $user): ?>
                            <tr>
                                <td class="ps-4">
                                    <div class="d-flex align-items-center">
                                        <div class="avatar avatar-sm me-3">
                                            <div class="avatar-initial bg-primary text-white rounded-circle d-flex align-items-center justify-content-center" style="width: 40px; height: 40px;">
                                                <?php echo strtoupper(substr($user['full_name'], 0, 1)); ?>
                                            </div>
                                        </div>
                                        <div>
                                            <div class="fw-semibold"><?php echo htmlspecialchars($user['full_name']); ?></div>
                                            <small class="text-muted">@<?php echo htmlspecialchars($user['username']); ?></small>
                                        </div>
                                    </div>
                                </td>
                                <td>
                                    <div class="text-truncate" style="max-width: 200px;" title="<?php echo htmlspecialchars($user['email']); ?>">
                                        <?php echo htmlspecialchars($user['email']); ?>
                                    </div>
                                </td>
                                <td>
                                    <span class="badge bg-<?php 
                                        echo $user['role'] === 'super_admin' ? 'warning' : 
                                             ($user['role'] === 'admin' ? 'primary' : 'secondary'); 
                                    ?>">
                                        <i class="fas fa-<?php 
                                            echo $user['role'] === 'super_admin' ? 'crown' : 
                                                 ($user['role'] === 'admin' ? 'user-shield' : 'user-gear'); 
                                        ?> me-1"></i>
                                        <?php echo htmlspecialchars(ucfirst(str_replace('_', ' ', $user['role']))); ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="badge bg-<?php echo $user['is_active'] ? 'success' : 'secondary'; ?>">
                                        <i class="fas fa-<?php echo $user['is_active'] ? 'check' : 'times'; ?> me-1"></i>
                                        <?php echo $user['is_active'] ? 'Active' : 'Inactive'; ?>
                                    </span>
                                    <?php if ($user['failed_attempts'] >= 3): ?>
                                        <span class="badge bg-danger ms-1">
                                            <i class="fas fa-lock me-1"></i>
                                            Locked
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <small class="text-muted">
                                        <?php 
                                            if ($user['last_login']) {
                                                $last_login = $user['last_login'];
                                                if ($last_login instanceof DateTime) {
                                                    echo $last_login->format('M j, Y H:i');
                                                } else {
                                                    echo date('M j, Y H:i', strtotime($last_login));
                                                }
                                            } else {
                                                echo '<span class="text-muted">Never</span>';
                                            }
                                        ?>
                                    </small>
                                </td>
                                <td>
                                    <small class="text-muted">
                                        <?php 
                                            $created = $user['created_at'];
                                            if ($created instanceof DateTime) {
                                                echo $created->format('M j, Y');
                                            } else {
                                                echo date('M j, Y', strtotime($created));
                                            }
                                        ?>
                                    </small>
                                </td>
                                <td class="text-end pe-4">
                                    <div class="btn-group btn-group-sm">
                                        <a href="?edit=<?php echo $user['id']; ?>" 
                                           class="btn btn-outline-primary" 
                                           data-bs-toggle="tooltip" 
                                           title="Edit User">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        
                                        <?php if ($user['failed_attempts'] > 0): ?>
                                        <form method="POST" class="d-inline">
                                            <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                                            <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                            <input type="hidden" name="action" value="reset_attempts">
                                            <button type="submit" class="btn btn-outline-warning" data-bs-toggle="tooltip" title="Reset Failed Attempts">
                                                <i class="fas fa-unlock"></i>
                                            </button>
                                        </form>
                                        <?php endif; ?>
                                        
                                        <?php if ($user['id'] != $_SESSION['admin_id']): ?>
                                        <form method="POST" class="d-inline">
                                            <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                                            <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                            <input type="hidden" name="action" value="toggle_status">
                                            <button type="submit" class="btn btn-<?php echo $user['is_active'] ? 'outline-warning' : 'outline-success'; ?>" 
                                                    data-bs-toggle="tooltip" 
                                                    title="<?php echo $user['is_active'] ? 'Deactivate User' : 'Activate User'; ?>">
                                                <i class="fas fa-<?php echo $user['is_active'] ? 'pause' : 'play'; ?>"></i>
                                            </button>
                                        </form>
                                        
                                        <form method="POST" class="d-inline" 
                                              onsubmit="return confirm('Are you sure you want to delete this user? This action cannot be undone.');">
                                            <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                                            <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                            <input type="hidden" name="action" value="delete_user">
                                            <button type="submit" class="btn btn-outline-danger" data-bs-toggle="tooltip" title="Delete User">
                                                <i class="fas fa-trash"></i>
                                            </button>
                                        </form>
                                        <?php else: ?>
                                        <span class="btn btn-outline-secondary disabled" data-bs-toggle="tooltip" title="Current User">
                                            <i class="fas fa-user"></i>
                                        </span>
                                        <?php endif; ?>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Add/Edit User Modal -->
<div class="modal fade" id="userModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">
                    <i class="fas fa-<?php echo $edit_user ? 'edit' : 'user-plus'; ?> me-2"></i>
                    <?php echo $edit_user ? 'Edit User' : 'Add New User'; ?>
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <form method="POST" action="">
                <div class="modal-body">
                    <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                    <input type="hidden" name="action" value="<?php echo $edit_user ? 'update_user' : 'create_user'; ?>">
                    <?php if ($edit_user): ?>
                        <input type="hidden" name="user_id" value="<?php echo $edit_user['id']; ?>">
                    <?php endif; ?>
                    
                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Full Name <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="full_name" 
                                   value="<?php echo $edit_user ? htmlspecialchars($edit_user['full_name']) : ''; ?>" 
                                   required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Username <span class="text-danger">*</span></label>
                            <input type="text" class="form-control" name="username" 
                                   value="<?php echo $edit_user ? htmlspecialchars($edit_user['username']) : ''; ?>" 
                                   required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Email <span class="text-danger">*</span></label>
                            <input type="email" class="form-control" name="email" 
                                   value="<?php echo $edit_user ? htmlspecialchars($edit_user['email']) : ''; ?>" 
                                   required>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Role <span class="text-danger">*</span></label>
                            <select class="form-select" name="role" required>
                                <option value="moderator" <?php echo ($edit_user && $edit_user['role'] === 'moderator') ? 'selected' : ''; ?>>Moderator</option>
                                <option value="admin" <?php echo ($edit_user && $edit_user['role'] === 'admin') ? 'selected' : ''; ?>>Admin</option>
                                <option value="super_admin" <?php echo ($edit_user && $edit_user['role'] === 'super_admin') ? 'selected' : ''; ?>>Super Admin</option>
                            </select>
                        </div>
                        <div class="col-12">
                            <label class="form-label">
                                Password <?php echo $edit_user ? '(leave blank to keep current)' : '<span class="text-danger">*</span>'; ?>
                            </label>
                            <input type="password" class="form-control" name="password" 
                                   <?php echo $edit_user ? '' : 'required'; ?>
                                   minlength="8"
                                   placeholder="<?php echo $edit_user ? 'Enter new password...' : 'Minimum 8 characters'; ?>">
                            <div class="form-text">
                                <i class="fas fa-info-circle me-1"></i>
                                Password must be at least 8 characters long
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fas fa-<?php echo $edit_user ? 'save' : 'plus'; ?> me-1"></i>
                        <?php echo $edit_user ? 'Update User' : 'Create User'; ?>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php if ($edit_user): ?>
<script>
    document.addEventListener('DOMContentLoaded', function() {
        var userModal = new bootstrap.Modal(document.getElementById('userModal'));
        userModal.show();
    });
</script>
<?php endif; ?>

<script>
    // Initialize tooltips
    document.addEventListener('DOMContentLoaded', function() {
        var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
        var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
            return new bootstrap.Tooltip(tooltipTriggerEl);
        });
    });
</script>